//
//  PredictionModels.swift
//  death_app Watch App
//
//  Data models for prediction engine
//

import Foundation

// MARK: - Core Prediction Models

struct LifeExpectancyPrediction: Codable {
    let lifeExpectancy: Double // Years
    let baseline: Double // Baseline without risk factors
    let riskFactors: RiskFactorBreakdown
    let combinedHazardRatio: Double
    let calculatedDate: Date
    let changeFromPrevious: PredictionChange?
}

struct RiskFactorBreakdown: Codable {
    let heartRate: Double
    let vo2Max: Double
    let smoking: Double
    let bmi: Double
    let sleep: Double
}

struct PredictionChange: Codable {
    let changeDays: Double // Change in days of life expectancy
    let changeDate: Date
    let previousDate: Date
    
    var isImprovement: Bool {
        return changeDays > 0
    }
    
    var formattedChange: String {
        let days = abs(changeDays)
        let sign = changeDays > 0 ? "+" : "-"
        
        if days < 1 {
            return "\(sign)< 1 day"
        } else if days < 365 {
            return "\(sign)\(Int(days)) days"
        } else {
            let years = days / 365.25
            return "\(sign)\(String(format: "%.1f", years)) years"
        }
    }
}

// MARK: - Health Data Models

struct PredictionHealthData: Hashable {
    let restingHeartRate: Double
    let vo2Max: Double
    let bmi: Double
    let averageSleepHours: Double
    
    // Additional health metrics that could be added
    let bloodPressureSystolic: Double?
    let bloodPressureDiastolic: Double?
    let cholesterolTotal: Double?
    let glucoseLevel: Double?
}

// MARK: - Demographics

public enum Gender: String, CaseIterable, Codable, Hashable {
    case male = "male"
    case female = "female"
    case other = "other"
}
